#include "config.h"

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "psutils.h"

#ifdef HAVE_LINUX_WIRELESS
// Some kernels include ethtool headers in wireless.h and seem to break
// terribly if these aren't defined
typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;
typedef unsigned long u64;

#include <asm/types.h>
#include <linux/if.h>
//#include <linux/wireless.h>
#endif

#include "util.h"

#if (defined(HAVE_LIBPCAP) && defined(SYS_LINUX) /*&& defined(HAVE_LINUX_WIRELESS) */)

#include "packetsourcetracker.h"
#include "packetsource_qca.h"
#include "qca_control.h"
#include "iwcontrol.h"

PacketSource_Qca::PacketSource_Qca(GlobalRegistry *in_globalreg, 
									 string in_interface,
									 vector<opt_pair> *in_opts) :
	PacketSource_Pcap(in_globalreg, in_interface, in_opts) {

		ParseOptions(in_opts);
		SetFCSBytes(4);
		SetValidateCRC(1);
}

int PacketSource_Qca::OpenSource() {
	int r = PacketSource_Pcap::OpenSource();
    const char *desc;

	if (r < 0)
		return r;

	if (DatalinkType() < 0) {
		if (pd != NULL)
			pcap_close(pd);
		return -1;
	}

    desc = pcap_datalink_val_to_description(datalink_type);
    _MSG("Datalink type: " + IntToString(datalink_type) + " " + "desc: " + string(desc), MSGFLAG_INFO);

	return 1;
}

int PacketSource_Qca::ParseOptions(vector<opt_pair> *in_opts) {
	PacketSource_Pcap::ParseOptions(in_opts);

	// Force us to keep the primary interface still running
	if (FetchOptBoolean("ignoreprimary", in_opts, false)) {
		ignore_primary_state = true;
		_MSG("Source '" + interface + "' will ignore the primary interface "
			 "up.  This may cause conflicts with wpasupplicant / networkmanager",
			 MSGFLAG_INFO);
	}

	if (FetchOpt("vap", in_opts) != "") {
		vap = FetchOpt("vap", in_opts);
		_MSG("Source '" + interface + "' will attempt to create a monitor-only "
			 "VIF '" + vap + "' instead of reconfiguring the main interface", 
			 MSGFLAG_INFO);
		// Opportunistic VAP off when specified
		opp_vap = 0;
	} else if (FetchOpt("vif", in_opts) != "") {
        vap = FetchOpt("vif", in_opts);
		_MSG("Source '" + interface + "' will attempt to create a monitor-only "
			 "VIF '" + vap + "' instead of reconfiguring the main interface", 
			 MSGFLAG_INFO);
        opp_vap = 0;
    }


	// Record if the VAP is absolutely forced (no passive blank option)
	if (FetchOptBoolean("forcevap", in_opts, 0))
		force_vap = 1;

    if (FetchOptBoolean("forcevif", in_opts, 0))
        force_vap = 1;

	if (FetchOptBoolean("fcsfail", in_opts, 0)) {
		if (vap == "") {
			_MSG("Source '" + interface + "': 'fcsfail' enabled to tell "
				 "mac80211 to report invalid packets, but not using a VIF. "
				 "A vif must be specified with 'vif=' in the source.", MSGFLAG_PRINTERROR);
		}
	}

	if (FetchOptBoolean("plcpfail", in_opts, 0)) {
		if (vap == "") {
			_MSG("Source '" + interface + "': 'plcpfail' enabled to tell "
				 "mac80211 to report invalid packets, but not using a VIF. "
				 "A vif must be specified with 'vif=' in the source", MSGFLAG_PRINTERROR);
		}
	}

	return 1;
}

int PacketSource_Qca::AutotypeProbe(string in_device) {
	string model;

	qca_find_board_type(model);

	if (model == "default") {
		type = "gwn";
	}
	else {
		type = model;
	}

	return 0;
}

int PacketSource_Qca::RegisterSources(Packetsourcetracker *tracker) {
	tracker->RegisterPacketProto("gwn7600-2g4", this, "IEEE80211b", 1);
	tracker->RegisterPacketProto("gwn7600-5g", this, "IEEE80211a", 1);
	tracker->RegisterPacketProto("gwn7610-2g4", this, "IEEE80211b", 1);
	tracker->RegisterPacketProto("gwn7610-5g", this, "IEEE80211a", 1);
	tracker->RegisterPacketProto("gwn7600lr-2g4", this, "IEEE80211b", 1);
	tracker->RegisterPacketProto("gwn7600lr-5g", this, "IEEE80211a", 1);
	tracker->RegisterPacketProto("gwn", this, "IEEE80211a", 1);

	return 1;
}

int PacketSource_Qca::EnableMonitor() {
	char errstr[STATUS_MAX];

    _MSG("interface '" + interface + "' " + "vap '" + vap + "'", MSGFLAG_INFO);
    // Try to grab the wireless mode before we go making vaps - don't make
	// a vap for an interface that is already in monitor mode.  ignore failures
	// and set a bogus stored mode so that we don't bypass the vap creation.  If
	// for some reason an interface doesn't exist but a vap can still be created
	// from it, we don't want to fall down
	if (Iwconfig_Get_Mode(vap.c_str(), errstr, &stored_mode) < 0) {
		_MSG(errstr, MSGFLAG_PRINTERROR);
		_MSG("Failed to get current wireless mode for interface '" + interface + 
			 "', check your configuration and consult the Kismet README file",
			 MSGFLAG_INFO);
	}

    if (qca_createvap(interface.c_str(), vap.c_str(), errstr) < 0) {
        _MSG("Source '" + interface + "' failed to create qca VIF(" + vap.c_str() + "): " +
                string(errstr), MSGFLAG_PRINTERROR);

        return -1;
    }

    // Switch our main processing interface to the vap
    interface = vap;

	// Try to grab the wireless mode
	if (Iwconfig_Get_Mode(interface.c_str(), errstr, &stored_mode) < 0) {
		_MSG(errstr, MSGFLAG_PRINTERROR);
		_MSG("Failed to get current wireless mode for interface '" + interface + 
			 "', check your configuration and consult the Kismet README file",
			 MSGFLAG_PRINTERROR);
		return -1;
	}

	// If it's already in monitor, make sure it's up and we're done
	if (stored_mode == IW_MODE_MONITOR) {
		_MSG("Interface '" + interface + "' is already marked as being in "
			 "monitor mode, leaving it as it is.", MSGFLAG_INFO);

		if (Ifconfig_Delta_Flags(interface.c_str(), errstr, 
								 IFF_UP | IFF_RUNNING | IFF_PROMISC) < 0) {
			_MSG(errstr, MSGFLAG_PRINTERROR);
			_MSG("Failed to bring up interface '" + interface + "', this "
				 "often means there is a problem with the driver (such as "
				 "missing firmware), check the output of `dmesg'.",
				 MSGFLAG_PRINTERROR);
			return -1;
		}
        sleep(1);

		return 0;
	}

	// Don't try this if we have a working rfmon interface, someone else 
	// probably wants the headers to stay as they are

	// Make sure it's up if nothing else
	return 0;
}

int PacketSource_Qca::DisableMonitor() {
	char errstr[STATUS_MAX];

	// We don't really care if any of these things fail.  Keep trying.
//	if (stored_channel > 0)
//		SetChannel(stored_channel);

	// We do care if this fails; reset the wireless mode if we need to and we're 
	// not a VAP (those get left, we don't care)
    if(qca_destroyvap(interface.c_str(), errstr) < 0) {
        _MSG("Qca source " + name + ": Failed to destroy VIF " +
             interface + " on shutdown: " +
             string(errstr), MSGFLAG_PRINTERROR);
        return -1;
    }
#if 0
	if (stored_mode > 0 && stored_mode != IEEE80211_M_MONITOR && vap == "") {
		if (Iwconfig_Set_Mode(interface.c_str(), errstr, stored_mode) < 0) {
			int oldflags;
			Ifconfig_Get_Flags(interface.c_str(), errstr, &oldflags);

			if (Ifconfig_Set_Flags(interface.c_str(), errstr,
								   oldflags & ~(IFF_UP | IFF_RUNNING)) < 0) {
				_MSG("Failed to restore previous wireless mode for interface '" +
					 interface + "'.  It may be left in an unknown or unusable state.",
					 MSGFLAG_PRINTERROR);
				return -1;
			}

			if (Iwconfig_Set_Mode(interface.c_str(), errstr, stored_mode) < 0) {
				_MSG("Failed to restore previous wireless mode for interface '" +
					 interface + "'.  It may be left in an unknown or unusable state.",
					 MSGFLAG_PRINTERROR);
				return -1;
			}
		}
	}

#endif
	return PACKSOURCE_UNMONITOR_RET_OKWITHWARN;
}

int PacketSource_Qca::SetChannel(unsigned int in_ch) {
	char errstr[STATUS_MAX];
	int err = 0;

	_MSG("Packet source '" + name + "' set channel " + IntToString(in_ch), MSGFLAG_INFO);
#if 1
	/* we use back ground scan to switch channels */
	err = bgscan_set_channel(interface.c_str(), in_ch, errstr);
	if (err >= 0) {
        consec_error = 0;
		return 1;
	}
#endif
#if 0
	if ((err = Iwconfig_Set_Channel(interface.c_str(), in_ch, errstr)) >= 0) {
		consec_error = 0;
		return 1;
	}
#endif
	_MSG(errstr, MSGFLAG_PRINTERROR);
	_MSG("Failed to change channel on source '" + name +"' and it looks "
		 "like the device has been removed (or the drivers have lost track of "
		 "it somehow)", MSGFLAG_ERROR);

	int curmode;
	if (Iwconfig_Get_Mode(interface.c_str(), errstr, &curmode) < 0) {
		_MSG(errstr, MSGFLAG_PRINTERROR);
		_MSG("Failed to change channel on source '" + name + "' and "
			 "failed to fetch current interface state when determining the "
			 "cause of the error.  It is likely that the drivers are in a "
			 "broken or unavailable state.", MSGFLAG_PRINTERROR);
		error = 1;
		return -1;
	}

	if (curmode != IEEE80211_M_MONITOR) {
		_MSG("Failed to change channel on source '" + name + "'. " 
			 "It appears to no longer be in monitor mode.  This can happen if "
			 "the drivers enter an unknown or broken state, but usually indicate "
			 "that an external program has changed the device mode.  Make sure no "
			 "network management tools (such as networkmanager) are running "
			 "before starting Kismet.", MSGFLAG_PRINTERROR);
		error = 1;
		return -1;
	}

	return 1;
}

vector<unsigned int> PacketSource_Qca::FetchSupportedChannels(string in_interface) {
	vector<unsigned int> ret;
	char errstr[STATUS_MAX];

	/* If we couldn't figure out what we are with mac80211, or if we don't
	 * have mac80211, go on to iwcontrol */
		/* I guess we don't really care about the return code here either */
    Iwconfig_Get_Chanlist(in_interface.c_str(), errstr, &ret);

	return ret;
}

int PacketSource_Qca::FetchHardwareChannel() {
    char errstr[STATUS_MAX] = "";
	int chan = 0;

    // Failure to fetch a channel isn't necessarily a fatal error
	// and if we blow up badly enough that we can't get channels, we'll
	// blow up definitively on something else soon enough
    if ((chan = Iwconfig_Get_Channel(interface.c_str(), errstr)) < 0) {
        globalreg->messagebus->InjectMessage("Source '" + name + "': " + errstr, 
											 MSGFLAG_INFO);
		chan = 0;
    }

	last_channel = chan;

    return chan;
}

#endif
