#include "config.h"

#if defined(HAVE_LIBNL20) || defined(HAVE_LIBNL30)
#define HAVE_LIBNL_NG
#endif

#ifdef SYS_LINUX

#include <dirent.h>
#include <fcntl.h>
#include <cstdio>
#include <cstring>
#include <cerrno>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <asm/types.h>
#include <linux/if.h>
#include <net/if_arp.h>
#include <string>
#include <vector>

#include "qca_control.h"


int qca_find_board_type(string &model) {
	string dirpath;
	FILE *procfs_data = NULL;
	char product_model[32];

	dirpath = "/proc/gxp/dev_info/dev_alias";

	procfs_data = fopen( dirpath.c_str(), "r" );
	if (procfs_data == NULL) {
		model = "default";
	}
	else {
		fgets( product_model, sizeof( product_model ), procfs_data );
		fclose( procfs_data );
		model = string(product_model);
	}

	return 0;
}

int qca_createvap(const char *interface, const char *newinterface, char *errstr) {
	char oname[IFNAMSIZ];
	int s;

	struct ieee80211_clone_params cp;
	struct ifreq ifr;

	memset(&ifr, 0, sizeof(ifr));
	memset(&cp, 0, sizeof(cp));
	memset(ifr.ifr_name, '\0', IFNAMSIZ);

	cp.icp_flags = IEEE80211_CLONE_BSSID;
	cp.icp_opmode = (u_int16_t)IEEE80211_M_MONITOR;
	strncpy(ifr.ifr_name, interface, sizeof(ifr.ifr_name));
	strncpy(cp.icp_name, newinterface, sizeof(cp.icp_name));
	ifr.ifr_data = (void *)&cp;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		snprintf(errstr, STATUS_MAX, "socket(SOCK_DRAGM)");
		return -1;
	}

	memset(oname, '\0', sizeof(oname));

	if (strlcpy(oname, ifr.ifr_name, sizeof(oname)) >= sizeof(oname)) {
		close(s);
		snprintf(errstr, STATUS_MAX, "VAP name too long\n");
		return -1;
	}
	if (ioctl(s, SIOC80211IFCREATE, &ifr) != 0) {
		snprintf(errstr, STATUS_MAX, "ioctl %d:%s", errno, strerror(errno));
		return -1;
	}
	/* NB: print name of clone device when generated */
	if (memcmp(oname, ifr.ifr_name, IFNAMSIZ) != 0)
		snprintf(errstr, STATUS_MAX, "%s\n", ifr.ifr_name);

	close(s);
	return 0;
}

int qca_destroyvap(const char *ifname, char *errstr)
{
	int s;
	struct ifreq ifr;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		snprintf(errstr, STATUS_MAX, "socket error");
		return -1;
	}

	memset(&ifr, 0, sizeof(ifr));
	if (strlcpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name)) >= sizeof(ifr.ifr_name)) {
		snprintf(errstr, STATUS_MAX, "ifname too long");
		close(s);
		return -1;
	}

	if (ioctl(s, SIOC80211IFDESTROY, &ifr) < 0)
		snprintf(errstr, STATUS_MAX, "ioctl %d:%s", errno, strerror(errno));
	close(s);

	return 0;
}

int bgscan_set_channel(const char *interface, int chan, char *errstr)
{
	int ret = 0;
	struct iwreq iwr;
	struct iw_scan_req req;

	int iosock = socket(PF_INET, SOCK_DGRAM, 0);
	if (iosock < 0) {
		snprintf(errstr, STATUS_MAX, "create socket failure\n");
		return -1;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, interface, IFNAMSIZ);

	if (0 != chan) {
		memset(&req, 0, sizeof(req));

		req.scan_type = IW_SCAN_TYPE_ACTIVE;
		req.bssid.sa_family = ARPHRD_ETHER;
		memset(req.bssid.sa_data, 0xff, ETH_ALEN);

		req.min_channel_time = KISMET_MIN_CHANNEL_TIME;
		req.max_channel_time = KISMET_MAX_CHANNEL_TIME;
		req.num_channels = 1;
		req.channel_list[0].m = chan;
		req.channel_list[0].e = 0;
		iwr.u.data.pointer = (caddr_t)&req;
		iwr.u.data.length = sizeof(struct iw_scan_req);
		iwr.u.data.flags = IEEE80211_IOC_GS_IDS_SCAN_REQ;
	}
	else {
		memset(&req, 0, sizeof(req));
		req.scan_type = IW_SCAN_TYPE_ACTIVE;
		req.min_channel_time = DFLT_MIN_CHANNEL_TIME;
		req.max_channel_time = DFLT_MAX_CHANNEL_TIME;
		iwr.u.data.pointer = (caddr_t)&req;
		iwr.u.data.length = sizeof(req);
		iwr.u.data.flags = IEEE80211_IOC_SCAN_REQ;
	}

	ret = ioctl(iosock, IEEE80211_IOCTL_P2P_BIG_PARAM, &iwr);
	close(iosock);

	return ret;
}

#endif
